// Figma 플러그인 메인 코드
figma.showUI(__html__, {
    width: 450,
    height: 700,
    title: "UX 라이팅 검사기"
});

// 선택된 텍스트 노드들을 저장할 변수
let selectedTextNodes = [];

// UI에서 메시지를 받았을 때 처리
figma.ui.onmessage = async (msg) => {
    try {
        switch (msg.type) {
            case 'get-selected-text':
                handleGetSelectedText();
                break;
            case 'get-full-report':
                await handleFullReport(msg.text, msg.apiKey, msg.rules);
                break;
            case 'apply-suggestion':
                await handleApplySuggestion(msg.nodeId, msg.newText);
                break;
            case 'save-rules':
                await handleSaveRules(msg.rules);
                break;
            case 'save-api-key':
                await handleSaveApiKey(msg.apiKey);
                break;
            case 'load-rules':
                await handleLoadRules();
                break;
            case 'load-api-key':
                await handleLoadApiKey();
                break;
        }
    } catch (error) {
        console.error('Error handling message:', error);
        figma.ui.postMessage({
            type: 'error',
            message: `오류가 발생했습니다: ${error.message}`
        });
    }
};

// 선택된 텍스트 노드 가져오기
function handleGetSelectedText() {
    const selection = figma.currentPage.selection;
    selectedTextNodes = [];
    
    // 선택된 노드들 중 텍스트 노드만 필터링
    function findTextNodes(nodes) {
        let textNodes = [];
        for (const node of nodes) {
            if (node.type === 'TEXT') {
                textNodes.push(node);
            } else if ('children' in node) {
                textNodes = textNodes.concat(findTextNodes(node.children));
            }
        }
        return textNodes;
    }
    
    const textNodes = findTextNodes(selection);
    
    if (textNodes.length === 0) {
        figma.ui.postMessage({
            type: 'no-text-selected',
            message: '텍스트 노드를 선택해주세요.'
        });
        return;
    }
    
    selectedTextNodes = textNodes;
    
    // 선택된 텍스트들을 UI로 전송
    const textData = textNodes.map(node => ({
        id: node.id,
        text: node.characters,
        name: node.name
    }));
    
    figma.ui.postMessage({
        type: 'text-selected',
        data: textData
    });
}

// 전체 리포트 생성
async function handleFullReport(text, apiKey, rules) {
    if (!apiKey) {
        figma.ui.postMessage({
            type: 'error',
            message: 'OpenAI API 키를 입력해주세요.'
        });
        return;
    }
    
    if (!rules || rules.length === 0) {
        figma.ui.postMessage({
            type: 'error',
            message: 'UX Writing 규칙을 하나 이상 설정해주세요.'
        });
        return;
    }
    
    try {
        figma.ui.postMessage({
            type: 'report-start',
            message: '텍스트를 분석하고 있습니다...'
        });

        // 여러 텍스트인 경우 병렬 처리
        if (selectedTextNodes.length > 1) {
            const promises = selectedTextNodes.map(async (node) => {
                const improvement = await getTextImprovement(node.characters, apiKey, rules);
                return {
                    original: node.characters,
                    ruleImproved: improvement.ruleImproved,
                    ruleViolations: improvement.ruleViolations,
                    uxImproved: improvement.uxImproved,
                    uxReason: improvement.uxReason
                };
            });
            
            const results = await Promise.all(promises);
            
            figma.ui.postMessage({
                type: 'ux-writing-result',
                multipleResults: true,
                results: results
            });
        } else {
            // 단일 텍스트 처리
            const improvement = await getTextImprovement(text, apiKey, rules);
            
            figma.ui.postMessage({
                type: 'ux-writing-result',
                singleResult: {
                    original: text,
                    ruleImproved: improvement.ruleImproved,
                    ruleViolations: improvement.ruleViolations,
                    uxImproved: improvement.uxImproved,
                    uxReason: improvement.uxReason
                }
            });
        }
        
    } catch (error) {
        console.error('Full report error:', error);
        figma.ui.postMessage({
            type: 'error',
            message: `분석 중 오류가 발생했습니다: ${error.message}`
        });
    }
}

// 텍스트 개선안 생성
async function getTextImprovement(text, apiKey, rules) {
    const rulesText = rules.map((rule, index) => `${index + 1}. ${rule}`).join('\n');
    
    const response = await fetch('https://api.openai.com/v1/chat/completions', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'Authorization': `Bearer ${apiKey}`
        },
        body: JSON.stringify({
            model: 'gpt-3.5-turbo',
            messages: [
                {
                    role: 'system',
                    content: `당신은 UX Writing 전문가입니다. 주어진 텍스트를 2가지 관점에서 분석해주세요:

1. 규칙 기반 검사: 주어진 규칙들을 엄격히 적용하여 검사
2. UX Writing 관점: 사용자 경험을 고려한 개선

검사 규칙:
${rulesText}

JSON 형식으로만 응답하세요:
{
  "ruleCheck": {
    "improved": "규칙 기반으로 수정된 텍스트",
    "violations": "위반된 규칙들과 구체적인 내용"
  },
  "uxImprovement": {
    "improved": "UX Writing 관점에서 개선된 텍스트",
    "reason": "UX 개선 이유"
  }
}`
                },
                {
                    role: 'user',
                    content: `다음 텍스트를 분석해주세요: "${text}"`
                }
            ],
            temperature: 0.1,
            max_tokens: 400
        })
    });
    
    if (!response.ok) {
        throw new Error(`OpenAI API 오류: ${response.status}`);
    }
    
    const apiResponse = await response.json();
    const resultContent = apiResponse.choices[0].message.content;
    
    try {
        const parsedResult = JSON.parse(resultContent);
        return {
            ruleImproved: parsedResult.ruleCheck.improved || text,
            ruleViolations: parsedResult.ruleCheck.violations || "분석 오류",
            uxImproved: parsedResult.uxImprovement.improved || text,
            uxReason: parsedResult.uxImprovement.reason || "분석 오류"
        };
    } catch (parseError) {
        return {
            ruleImproved: text,
            ruleViolations: "JSON 파싱 오류",
            uxImproved: text,
            uxReason: "JSON 파싱 오류"
        };
    }
}

// 제안된 텍스트 적용
async function handleApplySuggestion(nodeId, newText) {
    try {
        const node = figma.getNodeById(nodeId);
        if (node && node.type === 'TEXT') {
            await figma.loadFontAsync(node.fontName);
            node.characters = newText;
            figma.ui.postMessage({
                type: 'text-applied',
                message: '텍스트가 성공적으로 적용되었습니다.'
            });
        } else {
            figma.ui.postMessage({
                type: 'error',
                message: '텍스트 노드를 찾을 수 없습니다.'
            });
        }
    } catch (error) {
        figma.ui.postMessage({
            type: 'error',
            message: `텍스트 적용 중 오류가 발생했습니다: ${error.message}`
        });
    }
}

// 규칙 저장
async function handleSaveRules(rules) {
    try {
        await figma.clientStorage.setAsync('ux-writing-rules', rules);
    } catch (error) {
        console.error('Save rules error:', error);
    }
}

// API 키 저장
async function handleSaveApiKey(apiKey) {
    try {
        await figma.clientStorage.setAsync('openai-api-key', apiKey);
    } catch (error) {
        console.error('Save API key error:', error);
    }
}

// 규칙 불러오기
async function handleLoadRules() {
    try {
        const savedRules = await figma.clientStorage.getAsync('ux-writing-rules');
        figma.ui.postMessage({
            type: 'rules-loaded',
            rules: savedRules || []
        });
    } catch (error) {
        figma.ui.postMessage({
            type: 'error',
            message: '규칙 불러오기 중 오류가 발생했습니다.'
        });
    }
}

// API 키 불러오기
async function handleLoadApiKey() {
    try {
        const savedApiKey = await figma.clientStorage.getAsync('openai-api-key');
        figma.ui.postMessage({
            type: 'api-key-loaded',
            apiKey: savedApiKey || ''
        });
    } catch (error) {
        figma.ui.postMessage({
            type: 'error',
            message: 'API 키 불러오기 중 오류가 발생했습니다.'
        });
    }
}

// 플러그인 초기화
figma.on('selectionchange', () => {
    handleGetSelectedText();
});

// 초기 선택 상태 확인
handleGetSelectedText();
handleLoadRules();
handleLoadApiKey();